<?php

namespace App\Http\Controllers;

use App\Pages;
use Illuminate\Http\Request;
use App\Http\Requests\ValidationRequest;
use Illuminate\Support\Str;
use Carbon\Carbon;
use Helper;
use Image;
use File;
use Alert;
use Illuminate\Database\QueryException;
use Illuminate\Http\Response;
use Illuminate\Support\Facades\Storage;
use Yajra\DataTables\Facades\DataTables;

class PagesController extends Controller
{
    /**
     * Display a listing of the resource.
     *
     * @return \Illuminate\Http\Response
     */

    public function __construct()
    {
        $this->middleware('auth');
    }

    public function index()
    {
        return view('pages.be.halaman');
    }

    function json()
    {
        set_time_limit(0);
        ini_set('memory_limit', '-1');
        $data = Pages::orderBy('id', 'DESC')->get();
        return DataTables::of($data)
                    ->addIndexColumn()
                    ->editColumn('options', function($row){
                        $edit = route('pages.edit', $row->id);
                        $upload = route('pages.show', $row->id);
                        $btn = "&emsp;<a href='{$upload}' class='btn btn-sm btn-info modal-show' title='Form Upload Foto: {$row->name}'><i class='fa fa-upload'></i></a> <a href='{$edit}' class='btn btn-sm btn-warning modal-show edit' title='Form Edit Halaman: {$row->name}'><i class='fa fa-pencil'></i></a> <button type='button' class='btn btn-sm btn-danger' id='destroy' data-id='".$row->id."'><i class='fa fa-trash'></i></button>";
                        return $btn;
                    })
                    ->editColumn('status', function($row){
                        if($row->status == 1){
                            $status = "<label class='switch switch-primary'><input data-id='".$row->id."' type='checkbox' id='status' checked ><span></span></label>";
                        }else{
                            $status = "<label class='switch switch-primary'><input data-id='".$row->id."' type='checkbox' id='status'><span></span></label>";
                        }
                        return $status;
                    })
                    ->editColumn('created_at', function($row){
                        return $row->created_at != NULL ? \Helper::tgl_indo($row->created_at) : "-";
                    })
                    ->editColumn('updated_at', function($row){
                        return $row->updated_at != NULL ? \Helper::tgl_indo($row->updated_at) : "-";
                    })
                    ->rawColumns(['options', 'status'])
                    ->toJson();
    }

    /**
     * Show the form for creating a new resource.
     *
     * @return \Illuminate\Http\Response
     */
    public function create()
    {
        $model = new Pages();
        return view('pages.be.form_halaman', compact('model'));
    }

    /**
     * Store a newly created resource in storage.
     *
     * @param  \Illuminate\Http\Request  $request
     * @return \Illuminate\Http\Response
     */
    public function store(Request $request)
    {
        try {
            $this->validate($request, [
                'name' => 'required|string',
                'content' => 'required'
            ],[
                'name.required' => 'Nama Halaman harus diisi!',
                'name.string' => 'Hanya boleh string!',
                'content.required' => 'Konten harus diisi!'
            ]);

            $data = new Pages();
            $data->name = $request->name;
            $data->slug = Helper::createPageSlug($request->name);
            $data->content = $request->content;
            $data->created_at = Carbon::now('Asia/Jakarta');
            $data->updated_at = null;
            $data->save();
            \Helper::log(\Auth::user()->username, "Menambah halaman baru : ".$data->name);
            return response()->json(['code' => Response::HTTP_OK, 'msg' => 'Berhasil menambah halaman baru!']);
        } catch (QueryException $e) {
            return response()->json(['code' => Response::HTTP_UNPROCESSABLE_ENTITY, 'msg' => $e->errorInfo]);
        }
    }

    /**
     * Display the specified resource.
     *
     * @param  \App\Pages  $pages
     * @return \Illuminate\Http\Response
     */
    public function show($id)
    {
        $model = Pages::findOrFail($id);
        return view('pages.be.form_upload', compact('model'));
    }

    /**
     * Show the form for editing the specified resource.
     *
     * @param  \App\Pages  $pages
     * @return \Illuminate\Http\Response
     */
    public function edit($id)
    {
        $model = Pages::findOrFail($id);
        return view('pages.be.form_halaman', compact('model'));
    }

    /**
     * Update the specified resource in storage.
     *
     * @param  \Illuminate\Http\Request  $request
     * @param  \App\Pages  $pages
     * @return \Illuminate\Http\Response
     */
    public function update(Request $request, $id)
    {
        try {
            $this->validate($request, [
                'name' => 'required|string',
                'content' => 'required'
            ],[
                'name.required' => 'Nama Halaman harus diisi!',
                'name.string' => 'Hanya boleh string!',
                'content.required' => 'Konten harus diisi!'
            ]);

            $data = array(
                'name' => $request->name,
                'slug' => Helper::createPageSlug($request->name, $id),
                'content' => $request->content,
                'updated_at' => Carbon::now('Asia/Jakarta')
            );
            $pages = Pages::findOrFail($id);
            $pages->update($data);
            \Helper::log(\Auth::user()->username, "Mengupdate halaman : ".$pages->name);
            return response()->json(['code' => Response::HTTP_OK, 'msg' => 'Berhasil mengupdate halaman!']);
        } catch (QueryException $e) {
            return response()->json(['code' => Response::HTTP_UNPROCESSABLE_ENTITY, 'msg' => $e->errorInfo]);
        }

    }

    public function update_status(Request $request){
        try {
            $id = $request->id;
            $pages = Pages::findOrFail($id);
            if($pages->status == 0){
                $data = array('status' => '1', 'updated_at' => Carbon::now('Asia/Jakarta'));
            }else{
                $data = array('status' => '0', 'updated_at' => Carbon::now('Asia/Jakarta'));
            }
            $pages->update($data);
            \Helper::log(\Auth::user()->username, "Mengupdate status halaman : ".$pages->name);
            return response()->json(['code' => Response::HTTP_OK, 'msg' => 'Berhasil mengubah status halaman!']);
        } catch (QueryException $e) {
            return response()->json(['code' => Response::HTTP_UNPROCESSABLE_ENTITY, 'msg' => $e->errorInfo]);
        }
    }

    public function update_picture(ValidationRequest $request)
    {
        try {
            $id = $request->id;
            $pages = Pages::findOrFail($id);

            if(file_exists("assets/files/pages/".$pages->picture)){
                File::delete('assets/files/pages/'.$pages->picture);
                if(file_exists("assets/files/pages/thumbnail/".$pages->picture)){
                    File::delete('assets/files/pages/thumbnail/'.$pages->picture);
                }
            }
            if(!File::isDirectory('assets/files/pages')){
                File::makeDirectory('assets/files/pages', 0755, true, true);
            }
            if(!File::isDirectory('assets/files/pages/thumbnail')) {
                File::makeDirectory('assets/files/pages/thumbnail', 0755, true, true);
            }

            $file = $request->file('picture');
            $filename = strtolower('halaman-'.Helper::acakstring(8)).'.'.$request->picture->extension();
            $path = "assets/files/pages/".$filename;
            $uploaded = Helper::cropImage($file, $path, 800, 600);
            if($uploaded){
                $thumbnailsource = "assets/files/pages/".$filename;
                $thumbnailpath = "assets/files/pages/thumbnail/". $filename;
                Helper::createThumbnail($thumbnailsource, $thumbnailpath, 300, 225);
            }
            //create backupdata
            // Storage::disk('SFTP')->putFileAs(Helper::web_config('npsn')."/assets/files/pages/", $path, $filename);
            // Storage::disk('SFTP')->putFileAs(Helper::web_config('npsn')."/assets/files/pages/thumbnail", $thumbnailpath, $filename);
            $data = array(
                'picture' => $filename,
                'updated_at' => Carbon::now('Asia/Jakarta')
            );
            $pages->update($data);
            \Helper::log(\Auth::user()->username, "Mengupdate gambar halaman : ".$pages->name);
            return response()->json(['code' => Response::HTTP_OK, 'msg' => 'Berhasil mengganti gambar halaman!']);
        } catch (QueryException $e) {
            return response()->json(['code' => Response::HTTP_UNPROCESSABLE_ENTITY, 'msg' => $e->errorInfo]);
        }
    }

    public function hapus_picture(Request $request)
    {
        try {
            $id = $request->id;
            $pages = Pages::findOrFail($id);
            if(file_exists("assets/files/pages/".$pages->picture)){
                File::delete('assets/files/pages/'.$pages->picture);
                if(file_exists("assets/files/pages/thumbnail/".$pages->picture)){
                    File::delete('assets/files/pages/thumbnail/'.$pages->picture);
                }
            }
            $pages->update(['picture' => null, 'updated_at' => Carbon::now('Asia/Jakarta')]);
            \Helper::log(\Auth::user()->username, "Menghapus gambar halaman : ".$pages->name);
            return response()->json(['code' => Response::HTTP_OK, 'msg' => 'Berhasil menghapus gambar halaman!']);
        } catch (QueryException $e) {
            return response()->json(['code' => Response::HTTP_UNPROCESSABLE_ENTITY, 'msg' => $e->errorInfo]);
        }
    }

    /**
     * Remove the specified resource from storage.
     *
     * @param  \App\Pages  $pages
     * @return \Illuminate\Http\Response
     */
    public function destroy(Request $request)
    {
        try {
            $id = $request->input('id');
            $pages = Pages::findOrFail($id);
            if(file_exists("assets/files/pages/".$pages->picture)){
                File::delete('assets/files/pages/'.$pages->picture);
                if(file_exists("assets/files/pages/thumbnail/".$pages->picture)){
                    File::delete('assets/files/pages/thumbnail/'.$pages->picture);
                }
            }
            $pages->delete();
            \Helper::log(\Auth::user()->username, "Menghapus halaman : ".$pages->name);
            return response()->json(['code' => Response::HTTP_OK, 'msg' => 'Berhasil menghapus halaman!']);
        } catch (QueryException $e) {
            return response()->json(['code' => Response::HTTP_UNPROCESSABLE_ENTITY, 'msg' => $e->errorInfo]);
        }
    }
}
